<?php

namespace App\Core;

class Auth
{
    // JWT simples sem biblioteca externa
    public static function generateToken(array $payload): string
    {
        $config = require __DIR__ . '/../../config/config.php';
        $secret = $config['jwt']['secret'];
        
        $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
        $header = self::base64UrlEncode($header);
        
        $payload['exp'] = time() + $config['jwt']['expiration'];
        $payload = json_encode($payload);
        $payload = self::base64UrlEncode($payload);
        
        $signature = hash_hmac('sha256', "$header.$payload", $secret, true);
        $signature = self::base64UrlEncode($signature);
        
        return "$header.$payload.$signature";
    }
    
    public static function verifyToken(string $token): ?array
    {
        $config = require __DIR__ . '/../../config/config.php';
        $secret = $config['jwt']['secret'];
        
        $parts = explode('.', $token);
        if (count($parts) !== 3) {
            return null;
        }
        
        [$header, $payload, $signature] = $parts;
        
        $validSignature = self::base64UrlEncode(
            hash_hmac('sha256', "$header.$payload", $secret, true)
        );
        
        if ($signature !== $validSignature) {
            return null;
        }
        
        $payload = json_decode(self::base64UrlDecode($payload), true);
        
        if (!$payload || $payload['exp'] < time()) {
            return null;
        }
        
        return $payload;
    }
    
    public static function getCurrentUser(): ?array
    {
        $headers = getallheaders();
        if (!$headers) {
            $headers = self::getRequestHeaders();
        }
        
        $authHeader = $headers['Authorization'] ?? $headers['authorization'] ?? '';
        
        if (!preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
            return null;
        }
        
        return self::verifyToken($matches[1]);
    }
    
    public static function requireAuth(): array
    {
        $user = self::getCurrentUser();
        
        if (!$user) {
            http_response_code(401);
            echo json_encode(['error' => 'Unauthorized']);
            exit;
        }
        
        return $user;
    }
    
    public static function requireRole(string ...$roles): array
    {
        $user = self::requireAuth();
        
        if (!in_array($user['role'], $roles)) {
            http_response_code(403);
            echo json_encode(['error' => 'Forbidden']);
            exit;
        }
        
        return $user;
    }
    
    private static function base64UrlEncode(string $data): string
    {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }
    
    private static function base64UrlDecode(string $data): string
    {
        return base64_decode(strtr($data, '-_', '+/'));
    }
    
    private static function getRequestHeaders(): array
    {
        $headers = [];
        foreach ($_SERVER as $key => $value) {
            if (substr($key, 0, 5) === 'HTTP_') {
                $header = str_replace(' ', '-', ucwords(str_replace('_', ' ', strtolower(substr($key, 5)))));
                $headers[$header] = $value;
            }
        }
        return $headers;
    }
}
