<?php

namespace App\Controllers;

use App\Core\Database;
use App\Core\Auth;
use App\Core\Request;
use App\Core\Response;

class ProductController
{
    public function byRestaurant(string $restaurantId): string
    {
        $products = Database::query(
            'SELECT p.*, c.name as category_name 
             FROM products p 
             LEFT JOIN categories c ON p.category_id = c.id 
             WHERE p.restaurant_id = ? AND p.is_available = 1 
             ORDER BY c.display_order, p.name',
            [$restaurantId]
        );
        
        return Response::json($products);
    }
    
    public function create(): string
    {
        $user = Auth::requireRole('restaurant_owner');
        $data = Request::json();
        
        $restaurantId = $data['restaurant_id'] ?? 0;
        $categoryId = $data['category_id'] ?? null;
        $name = $data['name'] ?? '';
        $description = $data['description'] ?? '';
        $price = $data['price'] ?? 0;
        $imageUrl = $data['image_url'] ?? '';
        
        if (empty($name) || $price <= 0) {
            return Response::error('Dados inválidos');
        }
        
        Database::execute(
            'INSERT INTO products (restaurant_id, category_id, name, description, price, image_url) 
             VALUES (?, ?, ?, ?, ?, ?)',
            [$restaurantId, $categoryId, $name, $description, $price, $imageUrl]
        );
        
        $productId = Database::lastInsertId();
        
        $products = Database::query('SELECT * FROM products WHERE id = ?', [$productId]);
        
        return Response::json($products[0]);
    }
    
    public function update(string $id): string
    {
        Auth::requireRole('restaurant_owner');
        $data = Request::json();
        
        $fields = [];
        $params = [];
        
        if (isset($data['name'])) {
            $fields[] = 'name = ?';
            $params[] = $data['name'];
        }
        if (isset($data['description'])) {
            $fields[] = 'description = ?';
            $params[] = $data['description'];
        }
        if (isset($data['price'])) {
            $fields[] = 'price = ?';
            $params[] = $data['price'];
        }
        if (isset($data['is_available'])) {
            $fields[] = 'is_available = ?';
            $params[] = $data['is_available'];
        }
        
        if (empty($fields)) {
            return Response::error('Nenhum campo para atualizar');
        }
        
        $params[] = $id;
        
        $sql = 'UPDATE products SET ' . implode(', ', $fields) . ' WHERE id = ?';
        Database::execute($sql, $params);
        
        $products = Database::query('SELECT * FROM products WHERE id = ?', [$id]);
        
        return Response::json($products[0]);
    }
}
