<?php

namespace App\Controllers;

use App\Core\Database;
use App\Core\Auth;
use App\Core\Request;
use App\Core\Response;

class OrderController
{
    public function create(): string
    {
        $user = Auth::requireAuth();
        $data = Request::json();
        
        $restaurantId = $data['restaurant_id'] ?? 0;
        $items = $data['items'] ?? [];
        $deliveryAddressId = $data['delivery_address_id'] ?? 0;
        $paymentMethod = $data['payment_method'] ?? 'cash';
        $notes = $data['notes'] ?? '';
        
        if (empty($items)) {
            return Response::error('Carrinho vazio');
        }
        
        $db = Database::getConnection();
        
        try {
            $db->beginTransaction();
            
            // Calculate totals
            $subtotal = 0;
            foreach ($items as $item) {
                $products = Database::query('SELECT price FROM products WHERE id = ?', [$item['product_id']]);
                if (empty($products)) continue;
                $subtotal += $products[0]['price'] * $item['quantity'];
            }
            
            // Get delivery fee
            $restaurants = Database::query('SELECT delivery_fee FROM restaurants WHERE id = ?', [$restaurantId]);
            $deliveryFee = $restaurants[0]['delivery_fee'] ?? 0;
            $total = $subtotal + $deliveryFee;
            
            // Create order
            $orderNumber = 'ORD-' . time() . rand(1000, 9999);
            
            Database::execute(
                'INSERT INTO orders (order_number, customer_id, restaurant_id, delivery_address_id, 
                 subtotal, delivery_fee, total, payment_method, notes, status) 
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
                [$orderNumber, $user['userId'], $restaurantId, $deliveryAddressId, 
                 $subtotal, $deliveryFee, $total, $paymentMethod, $notes, 'pending']
            );
            
            $orderId = Database::lastInsertId();
            
            // Add order items
            foreach ($items as $item) {
                $products = Database::query('SELECT name, price FROM products WHERE id = ?', [$item['product_id']]);
                if (empty($products)) continue;
                
                $product = $products[0];
                $itemSubtotal = $product['price'] * $item['quantity'];
                
                Database::execute(
                    'INSERT INTO order_items (order_id, product_id, product_name, quantity, unit_price, subtotal, notes) 
                     VALUES (?, ?, ?, ?, ?, ?, ?)',
                    [$orderId, $item['product_id'], $product['name'], $item['quantity'], 
                     $product['price'], $itemSubtotal, $item['notes'] ?? '']
                );
            }
            
            $db->commit();
            
            $orders = Database::query('SELECT * FROM orders WHERE id = ?', [$orderId]);
            
            return Response::json($orders[0]);
            
        } catch (\Exception $e) {
            $db->rollBack();
            return Response::error('Erro ao criar pedido: ' . $e->getMessage(), 500);
        }
    }
    
    public function myOrders(): string
    {
        $user = Auth::requireAuth();
        
        $orders = Database::query(
            'SELECT o.*, r.name as restaurant_name, r.logo_url 
             FROM orders o 
             JOIN restaurants r ON o.restaurant_id = r.id 
             WHERE o.customer_id = ? 
             ORDER BY o.created_at DESC',
            [$user['userId']]
        );
        
        return Response::json($orders);
    }
    
    public function updateStatus(string $id): string
    {
        Auth::requireAuth();
        $data = Request::json();
        
        $status = $data['status'] ?? '';
        
        if (empty($status)) {
            return Response::error('Status é obrigatório');
        }
        
        Database::execute('UPDATE orders SET status = ? WHERE id = ?', [$status, $id]);
        
        $orders = Database::query('SELECT * FROM orders WHERE id = ?', [$id]);
        
        return Response::json($orders[0]);
    }
}
