<?php

namespace App\Controllers;

use App\Core\Database;
use App\Core\Auth;
use App\Core\Request;
use App\Core\Response;

class AuthController
{
    public function register(): string
    {
        $data = Request::json();
        
        $errors = Request::validate($data, [
            'email' => 'required|email',
            'password' => 'required|min:6',
            'name' => 'required'
        ]);
        
        if ($errors) {
            return Response::error(implode(', ', $errors));
        }
        
        $email = $data['email'];
        $password = $data['password'];
        $name = $data['name'];
        $phone = $data['phone'] ?? '';
        $role = $data['role'] ?? 'customer';
        
        // Check if email exists
        $existing = Database::query('SELECT id FROM users WHERE email = ?', [$email]);
        
        if ($existing) {
            return Response::error('Email já cadastrado');
        }
        
        // Create user
        $passwordHash = password_hash($password, PASSWORD_BCRYPT);
        
        Database::execute(
            'INSERT INTO users (email, password_hash, name, phone, role) VALUES (?, ?, ?, ?, ?)',
            [$email, $passwordHash, $name, $phone, $role]
        );
        
        $userId = Database::lastInsertId();
        
        $user = [
            'id' => $userId,
            'email' => $email,
            'name' => $name,
            'role' => $role,
        ];
        
        $token = Auth::generateToken(['userId' => $userId, 'role' => $role]);
        
        return Response::json(['user' => $user, 'token' => $token]);
    }
    
    public function login(): string
    {
        $data = Request::json();
        
        $email = $data['email'] ?? '';
        $password = $data['password'] ?? '';
        
        if (empty($email) || empty($password)) {
            return Response::error('Email e senha são obrigatórios');
        }
        
        $users = Database::query('SELECT * FROM users WHERE email = ?', [$email]);
        
        if (empty($users)) {
            return Response::unauthorized('Credenciais inválidas');
        }
        
        $user = $users[0];
        
        if (!password_verify($password, $user['password_hash'])) {
            return Response::unauthorized('Credenciais inválidas');
        }
        
        $token = Auth::generateToken(['userId' => $user['id'], 'role' => $user['role']]);
        
        unset($user['password_hash']);
        
        return Response::json(['user' => $user, 'token' => $token]);
    }
    
    public function me(): string
    {
        $currentUser = Auth::requireAuth();
        
        $users = Database::query(
            'SELECT id, email, name, phone, role, created_at FROM users WHERE id = ?',
            [$currentUser['userId']]
        );
        
        if (empty($users)) {
            return Response::notFound('Usuário não encontrado');
        }
        
        return Response::json($users[0]);
    }
}
